function CDX = import_cds_markit(directory)
% --------------------------------------------------------------------------------------------------
% Import the historical CDS data in the markit format.
% --------------------------------------------------------------------------------------------------
% directory         ... directory with CDS data from markit
% --------------------------------------------------------------------------------------------------
% sample call: import_cds_markit('c:/thesis/data/markit/cds data/')
% --------------------------------------------------------------------------------------------------

% Determine available dates
files = dir(directory);
file_names = {files(3:end).name};
ajd_dynamics = struct('k', 0.2, 'theta', 0.02, 'sigma', 0.01, 'L', 0.2, 'mu',0.03);

% Define empty CDX structure
CDX = struct( ...
    'dates', {{[] [] []}}, 'T', {{[] [] []}}, 'cut_offs', {[0.03 0.07 0.1 0.15 0.3]}, ...
    'index_market_price', {{[] [] []}}, ...
    'tranche_market_price_mid', {{[] [] []}}, ...
    'tranche_market_upfront_mid', {{[] [] []}}, ...
    'portfolio', [], ...
    'index_members', {repmat(NaN, length(10), 0)} ...
);

% Read all files in directory and merge their data with CDX structure
for i=1:length(file_names)
    % Import data
    disp(i)
    file_name = [directory file_names{i}];
    fid = fopen(file_name);
    m = textscan(fid, '%q');
    data = m{1}(7:end);
    fclose(fid);
    
    % Extract column names as well as position of desired CDS prices
    col_names = string_split(data{1});
    [trash, price_pos] = ismember({'Spread5y' 'Spread7y' 'Spread10y' 'Spread1y'}, col_names);
    [trash, rating_pos] = ismember({'AvRating'}, col_names);
    
    % Bring data into nice format
    data2 = {};
    for j=2:length(data)
        data2{j-1,1} = string_split(data{j});
    end
    
    % Extract names of CDS already existing in CDX structure
    if (length(CDX.portfolio) > 0)
        cds_names = {CDX.portfolio.company};
    else
        cds_names = {};
    end
    
    % Incorporate data into CDX structure
    for j=1:length(data2)
        % Check if seniority of debt is SNRFOR (=senior unsecured)
        seniority = data2{j}{5};
        if (~strcmp(seniority, 'SNRFOR'))
            continue
        end
        
        % Check if CDS name already exists, otherwise allocate memory
        ticker = data2{j}{2};
%         if (~strcmp('AT', ticker))
%             continue;
%         end
        cds_name = data2{j}{3};
        clip = data2{j}{4};
        if (~ismember(cds_name, cds_names))
            cds = struct( ...
                  'company', cds_name, 'ticker', ticker, 'clip', clip, 'dates', {{[] [] [] []}}, ...
                  'rating', {[]}, 'market_price', {{[] [] [] []}}, ...
                  'bid_ask_spread', {{[] [] [] []}}, 'x0', [], ...
                  'AJD', ajd_dynamics, 'LGD', 0.6, 'ai', 1, 'bi', 1);
            if (isempty(CDX.portfolio))
                CDX.portfolio = [cds];
            else
                CDX.portfolio(length(CDX.portfolio)+1) = cds;
            end
            cds_names{length(cds_names)+1} = cds_name;
        end
        
        % Determine position of CDS in portfolio
        [trash, pos] = ismember(cds_name, cds_names);
        
        % Extract rating and convert to numeric value (-1 = missing)
        rating = data2{j}(rating_pos);
        [found, rating] = ismember(rating, {'AAA' 'AA' 'A' 'BBB' 'BB' 'B'});
        if (found == 0)
            rating = -1;
        end
        
        % Insert new data into existing CDS price time series
        cds = CDX.portfolio(pos);
        new_date_num = datenum(data2{j}{1});
        prices = convert_prices(data2{j}(price_pos));
        % Insert prices
        for k=1:length(prices)  
            % Check if data available
            if (isnan(prices(k)))
                continue
            end
            
            % Determine position of new data in list of dates in order to have sorted time series
            if (length(cds.dates{k}) > 0)
                date_nums = cds.dates{k};
            else
                date_nums = [];
            end
            insert_pos = find((new_date_num > [0; date_nums]), 1, 'last');

            % Insert date end price into CDS structure
            cds.dates{k} = [cds.dates{k}(1:(insert_pos-1)); new_date_num; cds.dates{k}(insert_pos:end)];
            cds.market_price{k} = [cds.market_price{k}(1:(insert_pos-1)); prices(k); cds.market_price{k}(insert_pos:end)];
            % Currently no bid-ask spread available for cds data
            % cds.bid_ask_spread{k} = [cds.bid_ask_spread{k}(1:(insert_pos-1)); 0; cds.bid_ask_spread{k}(insert_pos:end)];
            
            % Insert rating
            if (k==1)
                cds.rating = [cds.rating(1:(insert_pos-1)); rating; cds.rating(insert_pos:end)];
            end
        end
        CDX.portfolio(pos) = cds;
    end
end

% Finally, for each CDS for each date, determine time to maturity (in years)
maturities = [5 7 10 1];
for i=1:length(CDX.portfolio)
    for k=1:length(maturities)
        % Determine matching IMM dates
        cds_dates = CDX.portfolio(i).dates{k};
        imm_dates = match_imm_dates(cds_dates, maturities(k));

        % Calculate time to maturity (in years)
        T = zeros(length(cds_dates),1);
        for j=1:length(cds_dates)
            T(j) = date_diff_years(datenum(cds_dates(j)), imm_dates(j));
        end
        
        % Insert data
        if (k==1)
            CDX.portfolio(i).T = {{[] [] [] []}};
        end
        CDX.portfolio(i).T{k} = T;
    end
end
